package jp.ill.photon.action;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.postgresql.util.PGobject;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.MetaObjectDao;
import jp.ill.photon.dao.MetaObjectDaoImpl;
import jp.ill.photon.util.DateUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.StringUtil;

public class ActionFactory {

	public static ActionFactory newInstance() {
		return new ActionFactory();
	}

	/**
	 * URLパターンに一致するAction設定を取得する
	 *
	 * @param requestURI
	 * @return
	 */
	public Map<String, Object> getActionSetting(String requestURL) {
		return getActionSetting(requestURL, false);
	}

	/**
	 * URLパターンに一致するAction設定を取得する
	 *
	 * @param requestURI
	 * @return
	 */
	public Map<String, Object> getActionSetting(String requestURL,
												boolean isError) {
		String targetPath = "";
		try {
			URL u = new URL(requestURL);
			targetPath = u.getPath();
		} catch (MalformedURLException e) {
			targetPath = requestURL;
		}
		Map<String, Object> aecAction = getActionByUrl(targetPath, isError);
		if (aecAction == null) {
			return null;
		}

		return buildActionSetting(aecAction);
	}

	public Map<String, Object> getActionSetting(String tenantId,
												String appId,
												String urlPattern) {

		Map<String, Object> aecAction = getActionByUrlPattern(tenantId, appId,
				urlPattern);
		if (aecAction == null) {
			return null;
		}

		return buildActionSetting(aecAction);
	}

	protected Map<String, Object> buildActionSetting(Map<String, Object> aecAction) {
		// Map<String, Object> setting = JsonUtil
		// .jsonToMap(((PGobject) aecAction.get("action_def")).getValue());

		Map<String, Object> setting = new HashMap<>();
		setting.put("tenant_id", (String) aecAction.get("tenant_id"));
		setting.put("app_id", (String) aecAction.get("app_id"));
		setting.put("app_dburl", (String) aecAction.get("dburl"));
		setting.put("app_dbuser", (String) aecAction.get("dbuser"));
		setting.put("app_dbpass", (String) aecAction.get("dbpass"));
		setting.put("action_id", (String) aecAction.get("action_id"));
		setting.put("name", (String) aecAction.get("name"));
		setting.put("remarks", (String) aecAction.get("remarks"));
		setting.put("url_root", (String) aecAction.get("url_root"));
		setting.put("url_combined", (String) aecAction.get("url_combined"));
		setting.put("url_save_flg", aecAction.get("url_save_flg"));
		setting.put("output_module_id",
				(String) aecAction.get("output_module_id"));
		setting.put("output_module_params", JsonUtil.jsonToMap(
				((PGobject) aecAction.get("output_module_params")).getValue()));
		setting.put("is_viewless_request",
				(String) aecAction.get("is_viewless_request"));
		setting.put("auth_level", (String) aecAction.get("auth_level"));
		setting.put("func_cd", (String) aecAction.get("func_cd"));
		setting.put("soft_const", JsonUtil.jsonToMap(
				((PGobject) aecAction.get("soft_const")).getValue()));
		setting.put("modules", JsonUtil
				.jsonToList(((PGobject) aecAction.get("modules")).getValue()));

		return setting;
	}

	public Map<String, Object> getInitParams(	HttpServletRequest request,
												Map<String, Object> setting) {

		String requestURL = request.getRequestURL().toString();
		Map<String, Object> params = getInitParams(requestURL, setting);

		if (request != null) {
			String urlWithQueryString = requestURL;
			String queryString = "";
			if (!StringUtils.isEmpty(request.getQueryString())) {
				queryString = "?" + request.getQueryString();
			}
			urlWithQueryString += queryString;
			params.put("_init.remote_address", request.getRemoteAddr());
			params.put("_init.remote_host", request.getRemoteHost());
			params.put("_init.method", request.getMethod());
			params.put("_init.header.referer", request.getHeader("REFERER"));
			params.put("_init.header.user_agent",
					request.getHeader("user-agent"));
			params.put("_init.url_with_query_string", urlWithQueryString);

			if (request.getSession() != null) {
				params.put("_init.session_id", request.getSession().getId());
			}
		}
		return params;
	}

	public Map<String, Object> getInitParams(	String requestURL,
												Map<String, Object> setting) {

		Map<String, Object> params = new HashMap<>();

		if (setting != null) {
			String tenantId = String.valueOf(setting.get("tenant_id"));

			Map<String, String> urlInfo = getUrlInfo(requestURL, setting);

			params.put("_init.timestamp", DateUtil.getYYYYMMDDHHMIMS());
			params.put("_init.tenant_id", tenantId);
			params.put("_init.app_id",
					StringUtil.defaultString(setting.get("app_id"), ""));
			params.put("_init.action_id",
					StringUtil.defaultString(setting.get("action_id"), ""));
			params.put("_init.auth_level",
					StringUtil.defaultString(setting.get("auth_level"), ""));
			params.put("_init.func_cd",
					StringUtil.defaultString(setting.get("func_cd"), ""));
			params.put("_init.url", urlInfo.get("url"));
			params.put("_init.url_save_flg",
					StringUtil.defaultString(setting.get("url_save_flg"), ""));
			params.put("_init.is_viewless_request", StringUtil
					.defaultString(setting.get("is_viewless_request"), ""));
			params.put("_init.base_url", urlInfo.get("base_url"));
			params.put("_init.app_url", urlInfo.get("app_url"));
			params.put("_init.context_url", urlInfo.get("context_url"));
			params.put("_init.servlet_url", urlInfo.get("servlet_url"));
		}

		return params;
	}

	public Map<String, Object> getInitParams(	String requestURL,
												Map<String, Object> setting,
												Map<String, Object> initParams) {
		Map<String, Object> params = getInitParams(requestURL, setting);
		params.put("_init.remote_address",
				initParams.getOrDefault("_init.remote_address", ""));
		params.put("_init.remote_host",
				initParams.getOrDefault("_init.remote_host", ""));
		params.put("_init.method", initParams.getOrDefault("_init.method", ""));
		params.put("_init.header.referer",
				initParams.getOrDefault("_init.header.referer", ""));
		params.put("_init.url_with_query_string", requestURL);
		return params;
	}

	/**
	 * リクエストURL情報を生成する
	 *
	 * @param requestURL
	 * @param actionSetting
	 * @return
	 */
	private Map<String, String> getUrlInfo(	String url,
											Map<String, Object> setting) {

		Map<String, String> urlInfo = new HashMap<>();
		StringBuffer u = new StringBuffer();

		try {
			URL tmpurl = new URL(url);
			u.append(tmpurl.getProtocol());
			u.append("://");
			u.append(tmpurl.getAuthority());
			urlInfo.put("base_url", u.toString());

		} catch (MalformedURLException e) {
			urlInfo.put("base_url", "");
		}

		urlInfo.put("url", url);
		if (setting != null) {
			u.append(setting.getOrDefault("url_root", "/"));
		}
		urlInfo.put("app_url", u.toString());
		urlInfo.put("context_url", u.toString().replaceAll("/[^/]+$", ""));

		urlInfo.put("servlet_url", url.replace(u.toString(), ""));
		return urlInfo;
	}

	/**
	 * 指定アプリのフック用アクションのマップ（キー=action_id）を返す。
	 *
	 * @param tenantId
	 * @param appId
	 * @return
	 */
	public Map<String, List<Map<String, Object>>> getFookActions(	String tenantId,
																	String appId) {

		Map<String, List<Map<String, Object>>> fookActionMap = new HashMap<>();

		MetaObjectDao dao = new MetaObjectDaoImpl();
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		List<Map<String, Object>> actionList = tm.required(() -> {
			return dao.selectFookActions(tenantId, appId);
		});

		if (actionList.size() > 0) {
			fookActionMap = actionList.stream()
					.collect(Collectors.toMap(
							s -> String.valueOf(s.get(
									"action_id")),
							s -> JsonUtil.jsonToList(
									((PGobject) s.get("modules")).getValue())));
		}

		return fookActionMap;
	}

	/**
	 * URLパターンに一致するAECActionを取得する
	 *
	 * @param requestURI
	 * @return
	 * @throws MalformedURLException
	 */
	protected Map<String, Object> getActionByUrl(	String targetPath,
													boolean isError) {
		MetaObjectDao dao = new MetaObjectDaoImpl();
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		List<Map<String, Object>> actionList = tm.required(() -> {
			return dao.selectActionByURL(targetPath, isError);
		});

		if (actionList.size() > 0) {
			return (Map<String, Object>) actionList.get(0);
		} else {
			return null;
		}
	}

	protected Map<String, Object> getActionByUrlPattern(String tenantId,
														String appId,
														String urlPattern) {
		MetaObjectDao dao = new MetaObjectDaoImpl();
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		List<Map<String, Object>> actionList = tm.required(() -> {
			return dao.selectActionByUrlPattern(tenantId, appId, urlPattern);
		});

		if (actionList.size() > 0) {
			return (Map<String, Object>) actionList.get(0);
		} else {
			return null;
		}
	}
}
