package jp.ill.photon.action;

import java.io.File;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonFrameworkException;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.exception.PhotonPageNotFoundException;
import jp.ill.photon.module.ModuleProcessor;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.util.DateUtil;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.StringUtil;

/**
 * AECのユーザーリクエストを処理するディスパッチャクラス
 *
 * @author h_tanaka
 *
 */
public class ActionDispatcher {

	public static ActionDispatcher getInstance() {
		return new ActionDispatcher();
	}

	/**
	 * URLを元にAECActionを選択してレスポンスデータを生成する
	 *
	 * @param requestURI
	 * @param parameters
	 * @param session
	 * @param remoteAddress
	 * @return
	 * @throws PhotonPageNotFoundException
	 */
	public ActionDto dispatch(	String requestURL,
								Map<String, Object> parameters,
								HttpSession session,
								String remoteAddress,
								Map<String, Object> dataMap,
								Map<String, Object> message)
			throws PhotonFrameworkException, PhotonModuleException,
			PhotonPageNotFoundException {

		return dispatch(requestURL, null, session, null, parameters, dataMap,
				message);

	}

	/**
	 * URLを元にAECActionを選択してレスポンスデータを生成する
	 *
	 * @param requestURI
	 * @param parameters
	 * @param session
	 * @param remoteAddress
	 * @return
	 * @throws PhotonPageNotFoundException
	 */
	public ActionDto dispatch(	String requestURL,
								Map<String, Object> initparameters,
								Map<String, Object> parameters,
								HttpSession session,
								String remoteAddress,
								Map<String, Object> dataMap,
								Map<String, Object> message)
			throws PhotonFrameworkException, PhotonModuleException,
			PhotonPageNotFoundException {

		return dispatch(requestURL, null, session, initparameters, parameters,
				dataMap, message);

	}

	/**
	 * URLを元にAECActionを選択してレスポンスデータを生成する
	 *
	 * @param requestURL
	 * @param parameters
	 * @param session
	 * @param remoteAddress
	 * @param dataMap
	 * @param request
	 * @return
	 */
	public ActionDto dispatch(	String requestURL,
								HttpServletRequest request,
								HttpSession session,
								Map<String, Object> initparameters,
								Map<String, Object> parameters,
								Map<String, Object> dataMap,
								Map<String, Object> message)
			throws PhotonFrameworkException, PhotonModuleException,
			PhotonPageNotFoundException {

		ActionFactory actInstance = ActionFactory.newInstance();

		Map<String, Object> actionSetting = actInstance
				.getActionSetting(requestURL);

		Map<String, Object> initparams = null;
		if (initparameters != null) {
			initparams = actInstance.getInitParams(requestURL, actionSetting,
					initparameters);
		} else if (request != null) {
			initparams = actInstance.getInitParams(request, actionSetting);
		} else {
			initparams = actInstance.getInitParams(requestURL, actionSetting);
		}

		return dispatchAction(requestURL, actionSetting, session, initparams,
				parameters, dataMap, message);
	}

	/**
	 * URL情報を元にアクションを実行する
	 *
	 * @param tenantId
	 * @param appId
	 * @param urlPattern
	 * @param initparameters
	 * @param parameters
	 * @param dataMap
	 * @param message
	 * @return
	 * @throws PhotonFrameworkException
	 * @throws PhotonModuleException
	 * @throws PhotonPageNotFoundException
	 */
	public ActionDto dispatch(	String tenantId,
								String appId,
								String urlPattern,
								Map<String, Object> initparameters,
								Map<String, Object> parameters,
								Map<String, Object> dataMap,
								Map<String, Object> message)
			throws PhotonFrameworkException, PhotonModuleException,
			PhotonPageNotFoundException {

		ActionFactory actInstance = ActionFactory.newInstance();

		Map<String, Object> actionSetting = actInstance
				.getActionSetting(tenantId, appId, urlPattern);

		Map<String, Object> initparams = null;
		if (initparameters != null) {
			initparams = actInstance.getInitParams(
					StringUtil.defaultString(
							actionSetting.getOrDefault("url_combined", "")),
					actionSetting, initparameters);
		} else {
			initparams = actInstance.getInitParams(
					StringUtil.defaultString(
							actionSetting.getOrDefault("url_combined", "")),
					actionSetting);
		}

		return dispatchAction(
				StringUtil.defaultString(
						actionSetting.getOrDefault("url_combined", "")),
				actionSetting, null, initparams, parameters, dataMap, message);
	}

	/**
	 * アクションメイン処理
	 *
	 * @param requestURL
	 * @param actionSetting
	 * @param session
	 * @param initparams
	 * @param parameters
	 * @param dataMap
	 * @param message
	 * @return
	 * @throws PhotonPageNotFoundException
	 * @throws PhotonFrameworkException
	 * @throws PhotonModuleException
	 */
	protected ActionDto dispatchAction(	String requestURL,
										Map<String, Object> actionSetting,
										HttpSession session,
										Map<String, Object> initparams,
										Map<String, Object> parameters,
										Map<String, Object> dataMap,
										Map<String, Object> message)
			throws PhotonPageNotFoundException, PhotonFrameworkException,
			PhotonModuleException {

		long sts = Long.parseLong(DateUtil.getYYYYMMDDHHMIMS());

		ActionDto dto = new ActionDto();
		dto.setAction(actionSetting);
		dto.setSession(session);
		dto.setInitParams(initparams);
		dto.setRawParams(parameters);
		if (dataMap != null) {
			dto.setDataMap(dataMap);
		}
		if (message != null) {
			dto.setMessages(message);
		}

		String actionName = "";
		if (actionSetting != null) {

			actionName = StringUtil
					.defaultString(actionSetting.get("action_id"), "");
			logger.info(String.format("[ACTION][%s] start", actionName));

			ActionParamMap actionParams = createActionParams(parameters);

			for (Entry<String, Object> param : initparams.entrySet()) {
				actionParams.put(param.getKey(), (String) param.getValue());
			}

			dto.setParams(actionParams);

			// パラメータやURLが変更されていれば、セッションにセットする
			if (session != null) {
				saveRequestUrlAndParams(actionSetting, parameters, session,
						requestURL);
			}

			dto = processActionModules(dto);

		} else {
			throw new PhotonPageNotFoundException("Actionが見つかりませんでした： "
					+ StringUtil.defaultString(requestURL, ""));
		}

		logger.info(String.format("[ACTION][%s] end %d ms", actionName,
				(Long.parseLong(DateUtil.getYYYYMMDDHHMIMS()) - sts)));
		return dto;
	}

	/**
	 * DTOに含まれるアクション情報を元にアクションのモジュールを実行して結果を返す
	 *
	 * @param dto
	 * @return
	 * @throws PhotonFrameworkException
	 * @throws PhotonModuleException
	 * @throws PhotonPageNotFoundException
	 */
	protected ActionDto processActionModules(ActionDto dto)
			throws PhotonFrameworkException, PhotonModuleException,
			PhotonPageNotFoundException {

		Map<String, List<Map<String, Object>>> fookActions = ActionFactory
				.newInstance()
				.getFookActions(dto.getTenantId(), dto.getAppId());

		// TODO #11348
		// DataSource ds = getAppDataSource(dto);
		// TransactionManager tm = getAppTransactionManager(ds);

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		final ActionDto moduleDto = dto;
		dto = tm.required(() -> {
			ActionDto resultDto = moduleDto;
			try {
				ModuleProcessor processor = new ModuleProcessor();
				resultDto = processor.processModules(moduleDto, fookActions);
			} catch (PhotonPageNotFoundException ex) {
				logger.error(logger.getStackTrace(ex));
				throw new RuntimeException(ex);
			} catch (PhotonModuleException ex) {
				logger.error(logger.getStackTrace(ex));
				throw new RuntimeException(ex);
			} catch (PhotonFrameworkException ex) {
				logger.error(logger.getStackTrace(ex));
				throw new RuntimeException(ex);
			} catch (Exception ex) {
				logger.error(logger.getStackTrace(ex));
				throw new RuntimeException(ex);
			}
			return resultDto;
		});

		if (dto.getResultType().equals("forward")) {
			String nextUrl = dto.getNextFullUrl();

			Map<String, Object> paramMap = dto.getRawParams();
			paramMap.putAll(dto.getAfterParams());

			dto = this.dispatch(nextUrl, null, dto.getSession(),
					dto.getInitParams(), paramMap, dto.getDataMap(),
					dto.getMessages());
		}

		// TODO どこかの標準モジュールに入れる
		dto.put("_app", "timeStamp", DateUtil.getYYYYMMDDHHMIMS());

		// return_typeが転送系ではない場合はOutputModule処理を通す
		if (Arrays.asList(new String[] { "continue", "break" })
				.contains(dto.getResultType()))

		{
			ModuleProcessor processor = new ModuleProcessor();
			dto = processor.processOutputModule(dto);
		}
		return dto;
	}

	/**
	 * アクションパラメータデータを作成する
	 *
	 * @param action
	 * @param parameters
	 * @return
	 */
	protected ActionParamMap createActionParams(Map<String, Object> parameters) {

		ActionParamMap aecParams = new ActionParamMap();

		// パラメータをString型配列として再生成する
		for (Map.Entry<String, Object> entry : parameters.entrySet()) {
			if (entry.getValue() instanceof File[]) {
				FileParam[] fileParams = createFileParam(entry.getKey(),
						(File[]) entry.getValue(), parameters);
				aecParams.getFileParams().put(entry.getKey(), fileParams);
			} else if (entry.getValue() instanceof String[]) {
				aecParams.getParams().put(entry.getKey(),
						(String[]) entry.getValue());
			} else {
				aecParams.put(entry.getKey(), (String) entry.getValue());
			}
		}

		return aecParams;
	}

	/**
	 * ファイルパラメータを扱いやすいようにオブジェクトのリストに変換する
	 *
	 * @param key
	 * @param files
	 * @param parameters
	 * @return
	 */
	protected FileParam[] createFileParam(	String key,
											File[] files,
											Map<String, Object> parameters) {
		String contentTypeKey = key + "ContentType";
		String[] contentTypes = (String[]) parameters
				.getOrDefault(contentTypeKey, new String[] {});
		int contentTypeSize = contentTypes.length;

		String fileNameKey = key + "FileName";
		String[] fileNames = (String[]) parameters.getOrDefault(fileNameKey,
				new String[] {});
		int fileNameSize = fileNames.length;

		List<FileParam> fileParams = new ArrayList<>();
		String contentType = null;
		String fileName = null;
		for (int i = 0; i < files.length; i++) {
			contentType = (contentTypeSize > i) ? contentTypes[i] : "";
			fileName = (fileNameSize > i) ? fileNames[i] : "";
			fileParams.add(new FileParam(files[i], contentType, fileName));
		}

		return fileParams.toArray(new FileParam[] {});
	}

	/**
	 * モジュールの戻り値変換設定に合わせてモジュールの戻り値を変換する 設定のないキーはモジュールの戻り値をそのまま返す
	 * パラメータタイプが1:の場合はtransfer_valをキー名として持つ値をreturn_keyのキー名で再設定する
	 * パラメータタイプが2:の場合はtransfer_valを固定値としてreturn_keyのキー名で設定する
	 *
	 * @param moduleReturnSettings
	 * @param result
	 * @return
	 */
	protected Map<String, Object> convertModuleReturns(	List<Map<String, Object>> moduleReturnSettings,
														ModuleResult result) {
		Map<String, Object> convertedResults = result.getReturnData();

		int transferType = 0;
		String returnKey = null;
		String transferVal = null;
		for (Map<String, Object> moduleReturnSetting : moduleReturnSettings) {
			transferType = (int) moduleReturnSetting.get("transfer_type");
			returnKey = (String) moduleReturnSetting.get("return_key");
			transferVal = (String) moduleReturnSetting.get("transfer_val");

			if (transferType == 1) {
				convertedResults.put(returnKey,
						convertedResults.get(transferVal));
			} else if (transferType == 2) {
				convertedResults.put(returnKey, transferVal);
			}
		}
		return convertedResults;
	}

	/***
	 * セッションをセットする
	 *
	 * @param session
	 * @param key
	 * @param tenantId
	 * @param value
	 */
	protected void setSessionByKeyAndTenantId(	HttpSession session,
												String key,
												String tenantId,
												String appId,
												Object value) {

		session.setAttribute(key + "-" + tenantId + "-" + appId, value);

	}

	/***
	 * セッションを取得する
	 *
	 * @param session
	 * @param key
	 * @param tenantId
	 * @return value
	 */
	protected Object getSessionByKeyAndTenantId(HttpSession session,
												String key,
												String tenantId,
												String appId) {

		return session.getAttribute(key + "-" + tenantId + "-" + appId);

	}

	/***
	 *
	 * パラメータやURLが変更されていれば、セッションにセットする
	 *
	 * @param tenantId
	 * @param appId
	 * @param parameters
	 * @param session
	 * @param requestURL
	 */
	@SuppressWarnings("unchecked")
	protected void saveRequestUrlAndParams(	Map<String, Object> actionSetting,
											Map<String, Object> parameters,
											HttpSession session,
											String requestURL) {
		String tenantId = StringUtil
				.defaultString(actionSetting.get("tenant_id"), "");
		String appId = StringUtil.defaultString(actionSetting.get("app_id"),
				"");

		String previousRequestUrl = (String) getSessionByKeyAndTenantId(session,
				"currentRequestUrl", tenantId, appId);
		Map<String, Object> previousRequestUrlParam = (Map<String, Object>) getSessionByKeyAndTenantId(
				session, "currentRequestUrlParam", tenantId, appId);

		boolean mapEquals = false;
		if (parameters == null) {
			if (previousRequestUrlParam == null) {
				mapEquals = true;
			}
		} else {
			if (previousRequestUrlParam != null) {

				// パラメータサイズの比較"
				if (parameters.size() != previousRequestUrlParam.size()) {
					// パラメータの数が違う
					mapEquals = false;
				} else {

					// パラメータの数は同じなので、parametersをループして検査
					mapEquals = true;
					for (Map.Entry<String, Object> currentParam : parameters
							.entrySet()) {

						// previousRequestUrlParamの中にキーが存在するか？");
						if (!previousRequestUrlParam
								.containsKey(currentParam.getKey())) {

							// 存在しないのでここでループ中断。パラメータの中身が変更されていると判定"
							mapEquals = false;
							break;

						}

						// 存在したのでここでループは継続
						// パラメータを配列に変換し、中身の精査を行う
						mapEquals = isSameParamObject(currentParam.getValue(),
								previousRequestUrlParam
										.get(currentParam.getKey()));

						if (!mapEquals) {
							// 配列の要素が変更されているので、ループ中断
							break;
						}

					}
				}
			}
		}

		// URLが前にアクセスしていたURLと違う、もしくは、URLパラメータが変更されていたとき
		if (!requestURL.equals(previousRequestUrl) || !mapEquals) {

			// URL、パラメータセット

			String currentRequestUrl = null;
			if (previousRequestUrl != null) {
				currentRequestUrl = new String(previousRequestUrl);
			}
			Map<String, Object> currentRequestUrlParam = null;
			if (previousRequestUrlParam != null) {
				currentRequestUrlParam = new HashMap<String, Object>(
						previousRequestUrlParam);
			}

			setSessionByKeyAndTenantId(session, "previousRequestUrl", tenantId,
					appId, currentRequestUrl);
			setSessionByKeyAndTenantId(session, "previousRequestUrlParam",
					tenantId, appId, currentRequestUrlParam);
			setSessionByKeyAndTenantId(session, "currentRequestUrl", tenantId,
					appId, requestURL);
			setSessionByKeyAndTenantId(session, "currentRequestUrlParam",
					tenantId, appId, parameters);

		}

	}

	/***
	 *
	 * 二つのパラメータの比較
	 *
	 * @param currentParamvalue
	 * @param previousParamvalue
	 * @return 同じかどうか（true:同じ / false:違う）
	 *
	 */
	@SuppressWarnings("rawtypes")
	private boolean isSameParamObject(	Object currentParamvalue,
										Object previousParamvalue) {

		if (currentParamvalue == null) {
			if (previousParamvalue == null) {
				return true;
			}
		} else {
			if (previousParamvalue != null) {

				Class currentParamvalueClass = currentParamvalue.getClass();
				Class previousParamvalueClass = previousParamvalue.getClass();

				// 型が同じか
				if (!currentParamvalueClass.getTypeName()
						.equals(currentParamvalueClass.getTypeName())) {
					return false;
				}

				// 長さが同じか
				int currentLength = 0;
				int previousLength = 0;
				if (currentParamvalueClass.isArray()
						&& previousParamvalueClass.isArray()) {
					currentLength = Array.getLength(currentParamvalue);
					previousLength = Array.getLength(previousParamvalue);
					if (currentLength != previousLength) {
						// 現在のパラメータと前回のパラメータのサイズが違う
						return false;
					}
				}

				// 現在のパラメータと前回のパラメータのサイズが同じ

				// 構成物が同じかをチェック
				if (currentParamvalueClass.isArray()
						&& previousParamvalueClass.isArray()) {

					// 配列のとき、要素をすべてみて、一つでも違っていればfalse
					for (int i = 0; i < currentLength; i++) {

						Object currentVal = Array.get(currentParamvalue, i);

						for (int j = 0; j < previousLength; j++) {

							Object previousVal = Array.get(previousParamvalue,
									j);

							if (currentVal == null) {
								if (previousVal != null) {
									// パラメータの構成要素が変更されている
									return false;
								}
							} else {
								if (!currentVal.equals(previousVal)) {
									// パラメータの構成要素が変更されている
									return false;
								}
							}

						}

					}

				} else {

					// 配列でないときは、equalsで確認するだけでよい
					if (!currentParamvalue.equals(previousParamvalue)) {
						return false;
					}

				}

			}

		}

		return true;

	}

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(ActionDispatcher.class);

}
